/*
 * Copyright (c) 2014,2016,2019-2020,2024-2025 LAAS/CNRS
 * All rights reserved.
 *
 * Redistribution  and  use  in  source  and binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of  source  code must retain the  above copyright
 *      notice and this list of conditions.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice and  this list of  conditions in the  documentation and/or
 *      other materials provided with the distribution.
 *
 * THE SOFTWARE  IS PROVIDED "AS IS"  AND THE AUTHOR  DISCLAIMS ALL WARRANTIES
 * WITH  REGARD   TO  THIS  SOFTWARE  INCLUDING  ALL   IMPLIED  WARRANTIES  OF
 * MERCHANTABILITY AND  FITNESS.  IN NO EVENT  SHALL THE AUTHOR  BE LIABLE FOR
 * ANY  SPECIAL, DIRECT,  INDIRECT, OR  CONSEQUENTIAL DAMAGES  OR  ANY DAMAGES
 * WHATSOEVER  RESULTING FROM  LOSS OF  USE, DATA  OR PROFITS,  WHETHER  IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR  OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *                                           Anthony Mallet on Fri May 23 2014
 */
#include "ac_tk3_flash.h"

#include <sys/types.h>
#include <sys/stat.h>

#include <err.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>

#include "flash.h"

static char *		elf_read_stdin(size_t *size);
static enum ldtype	elf_load_kind(const Elf32_Phdr *phdr, const char *raw);


/* --- elf_init ------------------------------------------------------------ */

/* Open ELF file, perform some consistency checks, and initialize internal
 * state
 */
Elf *
elf_init(const char *input, uint16_t machine)
{
  Elf32_Ehdr *ehdr;
  const char *id;
  int e;
  Elf *elf;
  size_t s;

  if (elf_version(EV_CURRENT) == EV_NONE) {
    warnx("%s", elf_errmsg(-1));
    return NULL;
  }

  if (strcmp(input, "-")) {
    /* open regular file */
    int elf_fd = open(input, O_RDONLY);
    if (elf_fd < 0) { warn("%s", input); return NULL; }

    if ((elf = elf_begin(elf_fd, ELF_C_READ, NULL)) == NULL) {
      e = elf_errno();
      if (!e)
        warnx("cannot load ELF file: %s", input);
      else
        warnx("%s: %s", input, elf_errmsg(e));
      return NULL;
    }
  } else {
    /* read stdin in memory */
    size_t len;
    char *data = elf_read_stdin(&len);
    if (!data) return NULL;

    if ((elf = elf_memory(data, len)) == NULL) {
      e = elf_errno();
      if (!e)
        warnx("cannot load ELF file: %s", input);
      else
        warnx("%s: %s", input, elf_errmsg(e));
      return NULL;
    }
  }

  if (elf_kind(elf) != ELF_K_ELF) {
    warnx("%s: wrong ELF format", input);
    return NULL;
  }

  /* architecture */
  id = elf_getident(elf, &s);
  if (!id) {
    warnx("%s: %s", input, elf_errmsg(-1));
    return NULL;
  }
  if (id[EI_CLASS] != ELFCLASS32 ||
      id[EI_DATA] != ELFDATA2LSB) {
    warnx("%s: wrong ELF format (expecting 32-bit LSB executable)", input);
    return NULL;
  }

  ehdr = elf32_getehdr(elf);
  if (!ehdr) {
    warnx("%s: cannot read ehdr: %s", input, elf_errmsg(-1));
    return NULL;
  }

  if (ehdr->e_type != ET_EXEC) {
    warnx("%s: wrong type (expecting executable file)", input);
    return NULL;
  }

  if (ehdr->e_machine != machine) {
    const char *machine_str;
    switch (machine) {
      case EM_ARM: machine_str = "ARM"; break;
      case EM_AVR: machine_str = "AVR"; break;

      default: warnx("%s: unsupported architecture", input); return NULL;
    }

    warnx("%s: wrong architecture (expecting %s)", input, machine_str);
    return NULL;
  }

  if (ehdr->e_phnum >= PN_XNUM) {
    warnx("%s: too many entries in the phdr", input);
    return NULL;
  }

  return elf;
}


/* --- elf_loadable_data --------------------------------------------------- */

char *
elf_loadable_data(Elf *elf, uintptr_t *paddr, size_t *size, enum ldtype kind)
{
  uintptr_t lopaddr, hipaddr;
  const Elf32_Ehdr *ehdr;
  const Elf32_Phdr *phdr;
  const char *raw;
  char *data;
  size_t n;
  int i;

  /* retrieve the program header table */
  ehdr = elf32_getehdr(elf);
  if (!ehdr) {
    warnx("cannot read extended header: %s", elf_errmsg(-1));
    return NULL;
  }

  phdr = elf32_getphdr(elf);
  if (!phdr) {
    warnx("cannot read program header: %s", elf_errmsg(-1));
    return NULL;
  }

  /* compute size of segments in the required paddr range */
  raw = elf_rawfile(elf, NULL);
  lopaddr = *paddr + *size;
  hipaddr = *paddr;
  for (i = 0; i < ehdr->e_phnum; i++) {
    if (phdr[i].p_type != PT_LOAD) continue;
    if (phdr[i].p_paddr < *paddr) continue;
    if (phdr[i].p_paddr >= *paddr + *size) continue;
    if (elf_load_kind(&phdr[i], raw) != kind) continue;

    if (lopaddr > phdr[i].p_paddr)
      lopaddr = phdr[i].p_paddr;
    if (hipaddr < phdr[i].p_paddr + phdr[i].p_memsz)
      hipaddr = phdr[i].p_paddr + phdr[i].p_memsz;

    printf("Found %s segment %d, paddr 0x%x-0x%x, %d bytes\n",
           kind == LD_DATA ? "data" : "tk3 settings", i,
           phdr[i].p_paddr, phdr[i].p_paddr + phdr[i].p_memsz - 1,
           phdr[i].p_memsz);
  }
  if (lopaddr < *paddr) lopaddr = *paddr;
  if (hipaddr > *paddr + *size) hipaddr = *paddr + *size;
  if (hipaddr <= lopaddr) {
    warnx("no %s in ELF file", kind == LD_DATA ? "data" : "tk3 settings");
    return NULL;
  }

  /* get data */
  data = malloc(hipaddr - lopaddr);
  if (!data) { warnx("out of memory"); return NULL; }
  memset(data, 0xff, hipaddr - lopaddr);

  for (i = 0; i < ehdr->e_phnum; i++) {
    if (phdr[i].p_type != PT_LOAD) continue;
    if (phdr[i].p_paddr < *paddr) continue;
    if (phdr[i].p_paddr >= *paddr + *size) continue;
    if (elf_load_kind(&phdr[i], raw) != kind) continue;

    n = phdr[i].p_filesz;
    if (phdr[i].p_paddr + phdr[i].p_filesz > hipaddr)
      n = hipaddr - phdr[i].p_paddr;

    memcpy(data + phdr[i].p_paddr - lopaddr, raw + phdr[i].p_offset, n);
  }

  *paddr = lopaddr;
  *size = hipaddr - lopaddr;
  return data;
}


/* --- elf_read_stdin ------------------------------------------------------- */

static char *
elf_read_stdin(size_t *size)
{
  size_t alloc, len;
  char *data, *p;
  ssize_t s;

  len = 0;
  alloc = 0;
  p = data = NULL;

  do {
    /* expand buffer */
    if (len == alloc) {
      if (!alloc) alloc = 4096; else alloc *= 2;
      p = realloc(data, alloc);
      if (!p) goto err;
      data = p;
      p = data + len;
    }

    /* read stdin */
    while(len < alloc) {
      s = read(0, p, alloc - len);
      if (s < 0) goto err;
      if (!s) break;
      p += s;
      len += s;
    }
  } while(s);

  /* fit buffer size */
  if (len < alloc) {
    p = realloc(data, len);
    if (!p) goto err;
    data = p;
  }

  *size = len;
  return data;

err:
  warn("-");
  if (data) free(data);
  return NULL;
}


/* --- elf_load_kind ------------------------------------------------------- */

static enum ldtype
elf_load_kind(const Elf32_Phdr *phdr, const char *raw)
{
  if (phdr->p_filesz < sizeof(TK3_MAGIC)) return LD_DATA;
  if (strcmp(raw + phdr->p_offset, TK3_MAGIC)) return LD_DATA;

  return LD_SETTINGS;
}
