//
// Copyright (c) 2018 - 2023, CNRS
// Authors: Joseph Mirabel, Florent Lamiraux
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:

// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.

// 2. Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following
// disclaimer in the documentation and/or other materials provided
// with the distribution.

// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
// FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
// COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
// OF THE POSSIBILITY OF SUCH DAMAGE.

#include <boost/python.hpp>
#include <boost/python/suite/indexing/vector_indexing_suite.hpp>
#include <hpp/core/collision-path-validation-report.hh>
#include <hpp/core/path-optimization/linear-constraint.hh>
#include <hpp/core/path-optimization/quadratic-program.hh>
#include <hpp/core/path-optimization/spline-gradient-based-abstract.hh>
#include <hpp/core/problem.hh>
#include <pinocchio/multibody/fwd.hpp>
#include <pyhpp/core/pathOptimization/fwd.hh>
#include <pyhpp/ref.hh>
#include <pyhpp/stl-pair.hh>
#include <pyhpp/util.hh>
#include <pyhpp/vector-indexing-suite.hh>

using namespace boost::python;

#define ADD_DATA_PROPERTY_READWRITE(TYPE, NAME, DOC) \
  def_readwrite(#NAME, &TYPE::NAME, DOC)

#define ADD_DATA_PROPERTY_READONLY_BYVALUE(TYPE, NAME, DOC)                    \
  add_property(                                                                \
      #NAME, make_getter(&TYPE::NAME, return_value_policy<return_by_value>()), \
      DOC)

namespace pyhpp {
namespace core {
namespace pathOptimization {
using namespace hpp::core;
using namespace hpp::core::pathOptimization;

template <int _PolynomeBasis, int _Order>
class SGBWrapper
    : public SplineGradientBasedAbstract<_PolynomeBasis, _Order>,
      public wrapper<SplineGradientBasedAbstract<_PolynomeBasis, _Order> > {
 public:
  typedef SplineGradientBasedAbstract<_PolynomeBasis, _Order> Base;
  typedef typename Base::Splines_t Splines_t;
  typedef typename Base::Reports_t Reports_t;
  typedef typename Base::SplineOptimizationData SplineOptimizationData;
  typedef typename Base::SplineOptimizationDatas_t SplineOptimizationDatas_t;

  // using Base::Base;
  SGBWrapper(const ProblemConstPtr_t& p) : Base(p) {}
  virtual ~SGBWrapper() {}

  PathVectorPtr_t optimize(const PathVectorPtr_t& path) {
    override f = this->get_override("optimize");
    if (!f) throw std::runtime_error("optimize not implemented in child class");
    // try {
    return f(path).as<PathVectorPtr_t>();
    // } catch (...) {
    // handle_exception();
    // return PathVectorPtr_t();
    // }
  }

  // The following functions are protected methods of
  // SplineGradientBasedAbstract. These wrappers are needed to make them
  // accessible.
  void appendEquivalentSpline(const PathVectorPtr_t& pv, Splines_t& ss) const {
    Base::appendEquivalentSpline(pv, ss);
  }

  void initializePathValidation(const Splines_t& splines) {
    override f = this->get_override("initializePathValidation");
    if (!f)
      Base::initializePathValidation(splines);
    else
      f(splines);
  }

  Reports_t validatePath(const Splines_t& splines,
                         std::vector<std::size_t>& reordering, bool stopAtFirst,
                         bool reorder) const {
    return Base::validatePath(splines, reordering, stopAtFirst, reorder);
  }

  void jointBoundConstraint(const Splines_t& splines,
                            LinearConstraint& lc) const {
    Base::jointBoundConstraint(splines, lc);
  }

  void addContinuityConstraints(const Splines_t& splines,
                                const size_type maxOrder,
                                const SplineOptimizationDatas_t& ess,
                                LinearConstraint& continuity) {
    Base::addContinuityConstraints(splines, maxOrder, ess, continuity);
  }

  PathVectorPtr_t buildPathVector(const Splines_t& splines) const {
    return Base::buildPathVector(splines);
  }

  void updateParameters(vectorRef_t param, const Splines_t& spline) const {
    vector_t _param(param);
    Base::updateParameters(_param, spline);
    param = _param;
  }
};

template <int _PolynomeBasis, int _Order>
void exposeSplineGradientBasedAbstract(const char* name) {
  typedef SplineGradientBasedAbstract<_PolynomeBasis, _Order> SGB_t;
  typedef SGBWrapper<_PolynomeBasis, _Order> SGBW_t;
  typedef hpp::shared_ptr<SGBW_t> Ptr_t;
  typedef typename SGB_t::Splines_t Splines_t;
  typedef typename SGBW_t::SplineOptimizationData SplineOptimizationData;
  typedef typename SGBW_t::SplineOptimizationDatas_t SplineOptimizationDatas_t;

  scope s = class_<SGBW_t, Ptr_t, bases<PathOptimizer>, boost::noncopyable>(
                name, init<const ProblemConstPtr_t&>())
                .PYHPP_DEFINE_METHOD(SGB_t, copy)
                .staticmethod("copy")
                .PYHPP_DEFINE_METHOD(SGB_t, updateSplines)
                .PYHPP_DEFINE_METHOD(SGBW_t, updateParameters)
                .PYHPP_DEFINE_METHOD(SGB_t, interpolate)
                .staticmethod("interpolate")
                .PYHPP_DEFINE_METHOD(SGBW_t, appendEquivalentSpline)
                .PYHPP_DEFINE_METHOD(SGBW_t, initializePathValidation)
                .def("validatePath", &SGBW_t::validatePath)
                .PYHPP_DEFINE_METHOD(SGBW_t, jointBoundConstraint)
                .PYHPP_DEFINE_METHOD(SGBW_t, addContinuityConstraints)
                .PYHPP_DEFINE_METHOD(SGBW_t, buildPathVector);

  class_<Splines_t>("Splines").def(
      cpp_like_vector_indexing_suite<Splines_t, true>());

  class_<SplineOptimizationData>("SplineOptimizationData", init<>())
      .def(init<size_type>())
      .def_readwrite("set", &SplineOptimizationData::set)
      .def_readwrite("es", &SplineOptimizationData::es)
      .def_readwrite("activeParameters",
                     &SplineOptimizationData::activeParameters);

  class_<SplineOptimizationDatas_t>("SplineOptimizationDatas"
                                    // )
                                    ,
                                    init<std::size_t>())
      .def(init<std::size_t, const SplineOptimizationData&>())
      // .def (vector_indexing_suite <SplineOptimizationDatas_t> ())
      .def("append", &vector_indexing_suite<SplineOptimizationDatas_t>::append)
      .def("__len__", &vector_indexing_suite<SplineOptimizationDatas_t>::size)
      .def("__getitem__",
           &vector_indexing_suite<SplineOptimizationDatas_t>::get_item,
           return_internal_reference<>())
      .def("__setitem__",
           &vector_indexing_suite<SplineOptimizationDatas_t>::set_item)
      // This is taken from
      // /usr/include/boost/python/suite/indexing/indexing_suite.hpp when #if
      // BOOST_WORKAROUND(BOOST_MSVC, < 1300) is false
      .def("__iter__",
           typename boost::mpl::if_<
               boost::mpl::bool_<false>, iterator<SplineOptimizationDatas_t>,
               iterator<SplineOptimizationDatas_t,
                        return_internal_reference<> > >::type());
}

void exposeNonTemplated() {
  using namespace hpp::core::path;

  typedef SGBWrapper<BernsteinBasis, 1> SGBW_t;
  typedef typename SGBW_t::Reports_t Reports_t;

  stl_pair<Reports_t::value_type::first_type,
           Reports_t::value_type::second_type>("Report");
  class_<Reports_t>("Reports").def(cpp_like_vector_indexing_suite<Reports_t>());
}

BOOST_PYTHON_MEMBER_FUNCTION_OVERLOADS(LC_reduceConstraint_overload,
                                       reduceConstraint, 2, 3)

void exposeSplineGradientBasedAbstracts() {
  using namespace hpp::core::path;
  exposeNonTemplated();
  exposeSplineGradientBasedAbstract<BernsteinBasis, 1>(
      "SplineGradientBasedAbstractB1");
  exposeSplineGradientBasedAbstract<BernsteinBasis, 3>(
      "SplineGradientBasedAbstractB3");
}

struct LCWrapper {
  static void setJ(LinearConstraint& lc, const matrix_t& J) { lc.J = J; }
  static void setb(LinearConstraint& lc, const vector_t& b) { lc.b = b; }
};

void exposeLinearConstraint() {
  class_<LinearConstraint>("LinearConstraint", init<size_type, size_type>())
      .PYHPP_DEFINE_METHOD(LinearConstraint, concatenate)
      .PYHPP_DEFINE_METHOD(LinearConstraint, decompose)
      .PYHPP_DEFINE_METHOD(LinearConstraint, computeRank)
      .def("reduceConstraint", &LinearConstraint::reduceConstraint,
           LC_reduceConstraint_overload(
               args("toReduce", "reduced", "computeRank")))
      .PYHPP_DEFINE_METHOD(LinearConstraint, computeSolution)
      .PYHPP_DEFINE_METHOD(LinearConstraint, isSatisfied)
      .PYHPP_DEFINE_METHOD(LinearConstraint, addRows)
      .add_property("J",
                    make_getter(&LinearConstraint::J,
                                return_value_policy<return_by_value>()),
                    &LCWrapper::setJ, "")
      .add_property("b",
                    make_getter(&LinearConstraint::b,
                                return_value_policy<return_by_value>()),
                    &LCWrapper::setb, "")
      .ADD_DATA_PROPERTY_READONLY_BYVALUE(LinearConstraint, rank, "")
      .ADD_DATA_PROPERTY_READONLY_BYVALUE(LinearConstraint, PK, "")
      .ADD_DATA_PROPERTY_READONLY_BYVALUE(LinearConstraint, xStar, "")
      .ADD_DATA_PROPERTY_READONLY_BYVALUE(LinearConstraint, xSol, "");
}

struct QPWrapper {
  static void setH(QuadraticProgram& qp, const matrix_t& H) { qp.H = H; }
  static void setb(QuadraticProgram& qp, const vector_t& b) { qp.b = b; }
};

void exposeQuadraticProblem() {
  class_<QuadraticProgram>("QuadraticProgram", init<size_type>())
      .def(init<const QuadraticProgram&, const LinearConstraint&>())
      .PYHPP_DEFINE_METHOD(QuadraticProgram, addRows)
      .PYHPP_DEFINE_METHOD(QuadraticProgram, reduced)
      .PYHPP_DEFINE_METHOD(QuadraticProgram, decompose)
      .def("solve",
           static_cast<void (QuadraticProgram::*)()>(&QuadraticProgram::solve))

      .PYHPP_DEFINE_METHOD(QuadraticProgram, computeLLT)
      .def("solve", static_cast<double (QuadraticProgram::*)(
                        const LinearConstraint&, const LinearConstraint&)>(
                        &QuadraticProgram::solve))

      .add_property("H",
                    make_getter(&QuadraticProgram::H,
                                return_value_policy<return_by_value>()),
                    &QPWrapper::setH, "")
      .add_property("b",
                    make_getter(&QuadraticProgram::b,
                                return_value_policy<return_by_value>()),
                    &QPWrapper::setb, "")
      .ADD_DATA_PROPERTY_READWRITE(QuadraticProgram, bIsZero, "")

      .ADD_DATA_PROPERTY_READONLY_BYVALUE(QuadraticProgram, llt, "")
      .ADD_DATA_PROPERTY_READONLY_BYVALUE(QuadraticProgram, trace, "")
      .ADD_DATA_PROPERTY_READONLY_BYVALUE(QuadraticProgram, activeConstraint,
                                          "")
      .ADD_DATA_PROPERTY_READONLY_BYVALUE(QuadraticProgram, activeSetSize, "")

      .ADD_DATA_PROPERTY_READONLY_BYVALUE(QuadraticProgram, dec, "")
      .ADD_DATA_PROPERTY_READONLY_BYVALUE(QuadraticProgram, xStar, "");
}
}  // namespace pathOptimization
}  // namespace core
}  // namespace pyhpp
