// Copyright (c) 2008-2014, Andrew Walker
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#include "dubins.hh"

#include <assert.h>
#include <math.h>

DubinsWord dubins_words[] = {
    dubins_LSL, dubins_LSR, dubins_RSL, dubins_RSR, dubins_RLR, dubins_LRL,
};

/**
 * Floating point modulus suitable for rings
 *
 * fmod doesn't behave correctly for angular quantities, this function does
 */
double fmodr(double x, double y) { return x - y * floor(x / y); }

double mod2pi(double theta) { return fmodr(theta, 2 * M_PI); }

int dubins_LSL(double alpha, double beta, double d, double* outputs) {
  UNPACK_INPUTS(alpha, beta);
  double tmp0 = d + sa - sb;
  double p_squared = 2 + (d * d) - (2 * c_ab) + (2 * d * (sa - sb));
  if (p_squared < 0) {
    return EDUBNOPATH;
  }
  double tmp1 = atan2((cb - ca), tmp0);
  double t = mod2pi(-alpha + tmp1);
  double p = sqrt(p_squared);
  double q = mod2pi(beta - tmp1);
  PACK_OUTPUTS(outputs);
  return EDUBOK;
}

int dubins_RSR(double alpha, double beta, double d, double* outputs) {
  UNPACK_INPUTS(alpha, beta);
  double tmp0 = d - sa + sb;
  double p_squared = 2 + (d * d) - (2 * c_ab) + (2 * d * (sb - sa));
  if (p_squared < 0) {
    return EDUBNOPATH;
  }
  double tmp1 = atan2((ca - cb), tmp0);
  double t = mod2pi(alpha - tmp1);
  double p = sqrt(p_squared);
  double q = mod2pi(-beta + tmp1);
  PACK_OUTPUTS(outputs);
  return EDUBOK;
}

int dubins_LSR(double alpha, double beta, double d, double* outputs) {
  UNPACK_INPUTS(alpha, beta);
  double p_squared = -2 + (d * d) + (2 * c_ab) + (2 * d * (sa + sb));
  if (p_squared < 0) {
    return EDUBNOPATH;
  }
  double p = sqrt(p_squared);
  double tmp2 = atan2((-ca - cb), (d + sa + sb)) - atan2(-2.0, p);
  double t = mod2pi(-alpha + tmp2);
  double q = mod2pi(-mod2pi(beta) + tmp2);
  PACK_OUTPUTS(outputs);
  return EDUBOK;
}

int dubins_RSL(double alpha, double beta, double d, double* outputs) {
  UNPACK_INPUTS(alpha, beta);
  double p_squared = (d * d) - 2 + (2 * c_ab) - (2 * d * (sa + sb));
  if (p_squared < 0) {
    return EDUBNOPATH;
  }
  double p = sqrt(p_squared);
  double tmp2 = atan2((ca + cb), (d - sa - sb)) - atan2(2.0, p);
  double t = mod2pi(alpha - tmp2);
  double q = mod2pi(beta - tmp2);
  PACK_OUTPUTS(outputs);
  return EDUBOK;
}

int dubins_RLR(double alpha, double beta, double d, double* outputs) {
  UNPACK_INPUTS(alpha, beta);
  double tmp_rlr = (6. - d * d + 2 * c_ab + 2 * d * (sa - sb)) / 8.;
  if (fabs(tmp_rlr) > 1) {
    return EDUBNOPATH;
  }
  double p = mod2pi(2 * M_PI - acos(tmp_rlr));
  double t = mod2pi(alpha - atan2(ca - cb, d - sa + sb) + mod2pi(p / 2.));
  double q = mod2pi(alpha - beta - t + mod2pi(p));
  PACK_OUTPUTS(outputs);
  return EDUBOK;
}

int dubins_LRL(double alpha, double beta, double d, double* outputs) {
  UNPACK_INPUTS(alpha, beta);
  double tmp_lrl = (6. - d * d + 2 * c_ab + 2 * d * (-sa + sb)) / 8.;
  if (fabs(tmp_lrl) > 1) {
    return EDUBNOPATH;
  }
  double p = mod2pi(2 * M_PI - acos(tmp_lrl));
  double t = mod2pi(-alpha - atan2(ca - cb, d + sa - sb) + p / 2.);
  double q = mod2pi(mod2pi(beta) - alpha - t + mod2pi(p));
  PACK_OUTPUTS(outputs);
  return EDUBOK;
}
